//-----------------------------------------------------------------------------
// Microsoft OLE DB RowsetViewer
// Copyright (C) 1994 - 1998 By Microsoft Corporation.
//
// @doc
//
// @module CTABLE.CPP
//
//-----------------------------------------------------------------------------------

//////////////////////////////////////////////////////////////////////////////
// Includes
//
//////////////////////////////////////////////////////////////////////////////
#include "Common.h"
#include "CTable.h"		
#include "msdaguid.h"		//CLSID_OLEDB_ROWPOSITIONLIBRARY


/////////////////////////////////////////////////////////////////
// CRowset::CRowset
//
/////////////////////////////////////////////////////////////////
CRowset::CRowset(CMDIChild* pCMDIChild, CDataSource* pCDataSource) : CBase(pCMDIChild)
{
	//DataSource
	ASSERT(pCDataSource);
	m_pCDataSource		= pCDataSource;
	m_pCSession			= new CSession(pCMDIChild, m_pCDataSource);
	m_pCCommand			= new CCommand(pCMDIChild, m_pCSession);
	m_pCRowPos			= new CRowPos(pCMDIChild);
	m_pCMultipleResults	= new CMultipleResults(pCMDIChild);
 	m_pCDataset			= new CDataset(pCMDIChild);
 
	//ColInfo
	m_cColumns		= 0;
	m_rgColumnInfo	= NULL;
	m_pStringBuffer	= NULL;

	//Rowset
	m_pIAccessor					= NULL;//Rowset Interface
	m_pIColumnsInfo					= NULL;//Rowset Interface
	m_pIConvertType					= NULL;//Rowset Interface
	m_pIRowset						= NULL;//Rowset Interface
	m_pIRowsetInfo					= NULL;//Rowset Interface
	m_pIColumnsRowset				= NULL;//Rowset Interface
	m_pIConnectionPointContainer	= NULL;//Rowset Interface
	m_pIDBAsynchStatus				= NULL;//Rowset Interface
	m_pIRowsetChange				= NULL;//Rowset Interface
	m_pIRowsetIdentity				= NULL;//Rowset Interface
	m_pIRowsetLocate				= NULL;//Rowset Interface
	m_pIRowsetFind					= NULL;//Rowset Interface
	m_pIRowsetView					= NULL;//Rowset Interface
	m_pIChapteredRowset				= NULL;//Rowset Interface
	m_pIRowsetResynch				= NULL;//Rowset Interface
	m_pIRowsetScroll				= NULL;//Rowset Interface
	m_pIRowsetUpdate				= NULL;//Rowset Interface
	m_pISupportErrorInfo			= NULL;//Rowset Interface

	//Extra Interfaces
	m_pIConnectionPoint				= NULL;//Connection Interface

	//Accessor
	m_cBindings		= 0;
	m_rgBindings    = NULL;
	m_hAccessor		= NULL;
	m_pData			= NULL;

	//Bookmark Accessor
	m_cBmkBindings	= 0;
	m_rgBmkBindings = NULL;
	m_hBmkAccessor	= NULL;
	m_fContainsBmk	= FALSE;

	//Listeners
	m_pCRowsetNotify	= new CRowsetNotify(m_pCMDIChild->m_pCListBox);
	m_pCAsynchNotify	= new CAsynchNotify(m_pCMDIChild->m_pCListBox);

	//Data
	m_ulEstimatedRows = 0;
}


/////////////////////////////////////////////////////////////////
// CRowset::~CRowset
//
/////////////////////////////////////////////////////////////////
CRowset::~CRowset()
{
	ReleaseRowset();
 	SAFE_DELETE(m_pCDataset);
 	SAFE_DELETE(m_pCMultipleResults);
	SAFE_DELETE(m_pCRowPos);
	SAFE_DELETE(m_pCCommand);
	SAFE_DELETE(m_pCSession);
	SAFE_DELETE(m_pCDataSource);

	//Verify Listeners have no extra references on them...
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	pCListBox->OutputRelease((IUnknown**)&m_pCRowsetNotify, "IRowsetNotify", 0);
	pCListBox->OutputRelease((IUnknown**)&m_pCAsynchNotify, "IDBAsynchNotify", 0);
}


/////////////////////////////////////////////////////////////////
// HRESULT CRowset::SetInterface
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::SetInterface(REFIID riid, IUnknown* pIUnknown)
{
	HRESULT hr = S_OK;

	//no-op
	if(pIUnknown == NULL)
		return E_INVALIDARG;

	//If the requested interface is a Rowset interface, place it in the
	//Appropiate member variable, otherwise we have to release it, since we
	//have no place to store it...
	if(riid == IID_IUnknown) 
		m_pIUnknown = (IUnknown*)pIUnknown;
	else if(riid == IID_IAccessor) 
		m_pIAccessor = (IAccessor*)pIUnknown;
	else if(riid == IID_IColumnsInfo) 
		m_pIColumnsInfo = (IColumnsInfo*)pIUnknown;
	else if(riid == IID_IConvertType) 
		m_pIConvertType = (IConvertType*)pIUnknown;
	else if(riid == IID_IRowset) 
		m_pIRowset = (IRowset*)pIUnknown;
	else if(riid == IID_IRowsetInfo) 
		m_pIRowsetInfo = (IRowsetInfo*)pIUnknown;
	else if(riid == IID_IColumnsRowset) 
		m_pIColumnsRowset = (IColumnsRowset*)pIUnknown;
	else if(riid == IID_IConnectionPointContainer) 
		m_pIConnectionPointContainer = (IConnectionPointContainer*)pIUnknown;
	else if(riid == IID_IDBAsynchStatus) 
		m_pIDBAsynchStatus = (IDBAsynchStatus*)pIUnknown;
	else if(riid == IID_IRowsetChange) 
		m_pIRowsetChange = (IRowsetChange*)pIUnknown;
	else if(riid == IID_IRowsetIdentity) 
		m_pIRowsetIdentity = (IRowsetIdentity*)pIUnknown;
	else if(riid == IID_IRowsetLocate) 
		m_pIRowsetLocate = (IRowsetLocate*)pIUnknown;
	else if(riid == IID_IRowsetFind) 
		m_pIRowsetFind = (IRowsetFind*)pIUnknown;
	else if(riid == IID_IRowsetView) 
		m_pIRowsetView = (IRowsetView*)pIUnknown;
	else if(riid == IID_IChapteredRowset) 
		m_pIChapteredRowset = (IChapteredRowset*)pIUnknown;
	else if(riid == IID_IRowsetResynch) 
		m_pIRowsetResynch = (IRowsetResynch*)pIUnknown;
	else if(riid == IID_IRowsetScroll) 
		m_pIRowsetScroll = (IRowsetScroll*)pIUnknown;
	else if(riid == IID_IRowsetUpdate) 
		m_pIRowsetUpdate = (IRowsetUpdate*)pIUnknown;
	else if(riid == IID_ISupportErrorInfo) 
		m_pISupportErrorInfo = (ISupportErrorInfo*)pIUnknown;
	else
		hr = E_NOINTERFACE;

	return hr;
}


/////////////////////////////////////////////////////////////////
// CRowset::ReleaseRowset
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::ReleaseRowset()
{
	//ColInfo
	SAFE_FREE(m_rgColumnInfo);
	SAFE_FREE(m_pStringBuffer);

	ReleaseAccessor(&m_hAccessor);
	FreeBindings(&m_cBindings, &m_rgBindings);
	SAFE_FREE(m_pData);

	m_fContainsBmk = FALSE;
	ReleaseAccessor(&m_hBmkAccessor);
	FreeBindings(&m_cBmkBindings, &m_rgBmkBindings);

	//Rowset
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	pCListBox->OutputRelease((IUnknown**)&m_pIAccessor,			"IAccessor");
	pCListBox->OutputRelease((IUnknown**)&m_pIColumnsInfo,		"IColumnsInfo");
	pCListBox->OutputRelease((IUnknown**)&m_pIConvertType,		"IConvertType");
	pCListBox->OutputRelease((IUnknown**)&m_pIRowset,			"IRowset");
	pCListBox->OutputRelease((IUnknown**)&m_pIRowsetInfo,		"IRowsetInfo");
	pCListBox->OutputRelease((IUnknown**)&m_pIColumnsRowset,	"IColumnsRowset");
	pCListBox->OutputRelease((IUnknown**)&m_pIConnectionPointContainer, "IConnectionPointContainer");
	pCListBox->OutputRelease((IUnknown**)&m_pIDBAsynchStatus,	"IDBAsynchStatus");
	pCListBox->OutputRelease((IUnknown**)&m_pIRowsetChange,		"IRowsetChange");
	pCListBox->OutputRelease((IUnknown**)&m_pIRowsetIdentity,	"IRowsetIdentity");
	pCListBox->OutputRelease((IUnknown**)&m_pIRowsetLocate,		"IRowsetLocate");
	pCListBox->OutputRelease((IUnknown**)&m_pIRowsetFind,		"IRowsetFind");
	pCListBox->OutputRelease((IUnknown**)&m_pIRowsetView,		"IRowsetView");
	pCListBox->OutputRelease((IUnknown**)&m_pIChapteredRowset,	"IChapteredRowset");
	pCListBox->OutputRelease((IUnknown**)&m_pIRowsetResynch,	"IRowsetResynch");
	pCListBox->OutputRelease((IUnknown**)&m_pIRowsetScroll,		"IRowsetScroll");
	pCListBox->OutputRelease((IUnknown**)&m_pIRowsetUpdate,		"IRowsetUpdate");
	pCListBox->OutputRelease((IUnknown**)&m_pISupportErrorInfo,	"IRowsetErrorInfo");

	//Extra intefaces
	pCListBox->OutputRelease((IUnknown**)&m_pIConnectionPoint,	"IConnectionPoint");

	//RowPosition
	m_pCRowPos->ReleaseRowPos();

	//Base
	ReleaseBase("Rowset");

	//Don't Unadvise the Lsiteners until the Rowset goes away.
	//We want to receive and Display the ROWSET_RELEASE notifications
	UnadviseListeners();
	return S_OK;
}


////////////////////////////////////////////////////////////////
// CRowset::CreateConnection
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::CreateConnection(IUnknown* pIUnknown, ULONG cPropSets, DBPROPSET* rgPropSets, DWORD dwConnectOptions)
{
	HRESULT hr = E_FAIL;

	//DataSource - Connect using the passed in Instance
	TESTC(hr = m_pCDataSource->Connect(pIUnknown));
	
	//DataSource - SetProperties
	if(dwConnectOptions & CONNECT_SETPROPERTIES)
		TESTC(hr = m_pCDataSource->SetProperties(cPropSets, rgPropSets));

	//DataSource - Initialize
	if(dwConnectOptions & CONNECT_INITIALIZE)
	{
		//Initialize and Obtain datasource interfaces
		TESTC(hr = m_pCDataSource->Initialize());
	}
	else
	{
		//Just obtain whatever datasource interfaces are avialable...
		TESTC(hr = m_pCDataSource->CreateDataSource(m_pCDataSource));
	}

	//Enumerators don't have sessions or commands
	if(!m_pCDataSource->m_pISourcesRowset)
	{
		//Session 
		if(dwConnectOptions & CONNECT_CREATESESSION)
			TESTC(hr = m_pCSession->CreateSession());

		//Command (optional - provider may not support commands)
		if(dwConnectOptions & CONNECT_CREATECOMMAND)
			m_pCCommand->CreateCommand();
	}

CLEANUP:
	return hr;
}


////////////////////////////////////////////////////////////////
// CRowset::CreateConnection
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::CreateConnection(CRowset* pCRowset, BOOL bNewSession)
{
	ASSERT(pCRowset);
	HRESULT hr = E_FAIL;

	//DataSource - Connect using the passed in Instance and Properties...
	TESTC(hr = m_pCDataSource->CreateDataSource(pCRowset->m_pCDataSource));
	
	//Enumerators don't have sessions or commands
	if(!m_pCDataSource->m_pISourcesRowset)
	{
		//Session 
		if(pCRowset->m_pCSession->m_pIUnknown)
		{
			if(bNewSession)
			{
				TESTC(hr = m_pCSession->CreateSession());
			}
			else
			{
				TESTC(hr = m_pCSession->CreateSession(pCRowset->m_pCSession));
			}
		}

		//Command (optional - provider may not support commands)
		if(pCRowset->m_pCCommand->m_pIUnknown)
			m_pCCommand->CreateCommand();
	}

CLEANUP:
	return hr;
}

/////////////////////////////////////////////////////////////////
// HRESULT CRowset::AdviseListeners
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::AdviseListeners()
{
	HRESULT hr = E_FAIL;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	
	//IRowsetNotify Listener
	//This is optional functionality so don't post error
	if(m_pIUnknown && GetOptionsObj()->m_dwNotifyOpts & NOTIFY_IROWSETNOTIFY)
		hr = m_pCRowsetNotify->Advise(m_pIUnknown); 

	//IDBAsynchNotify Listener
	//This is optional functionality so don't post error
	if(m_pIUnknown && GetOptionsObj()->m_dwNotifyOpts & NOTIFY_IDBASYNCHNOTIFY)
		hr = m_pCAsynchNotify->Advise(m_pIUnknown); 

	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CRowset::UnadviseListeners
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::UnadviseListeners()
{
	HRESULT hr = E_FAIL;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	
	//IRowsetNotify Listener
	//This is optional functionality so don't post error
	if(m_pIUnknown)
		hr = m_pCRowsetNotify->Unadvise(m_pIUnknown);

	//IDBAsynchNotify Listener
	//This is optional functionality so don't post error
	if(m_pIUnknown)
		hr = m_pCAsynchNotify->Unadvise(m_pIUnknown); 

	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CRowset::GetColInfo
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::GetColInfo(IColumnsInfo* pIColumnsInfo)
{
	HRESULT		hr = E_FAIL;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	
	//Release an Previous Info
	SAFE_FREE(m_rgColumnInfo);
	SAFE_FREE(m_pStringBuffer);

	//Default this this objects IColumnsInfo
	if(pIColumnsInfo == NULL)
		pIColumnsInfo = m_pIColumnsInfo;

	if(pIColumnsInfo)
	{
		//Get column information from the command object via IColumnsInfo::GetColumnsInfo 
		TESTC(pCListBox->OutputPreMethod("IColumnsInfo::GetColumnInfo(&%d, &0x%08x, &0x%08x)", m_cColumns, m_rgColumnInfo, m_pStringBuffer));
		XTEST(hWnd, hr = pIColumnsInfo->GetColumnInfo(&m_cColumns, &m_rgColumnInfo, &m_pStringBuffer));
		TESTC(pCListBox->OutputPostMethod(hr, "IColumnsInfo::GetColumnInfo(&%d, &0x%08x, &0x%08x)", m_cColumns, m_rgColumnInfo, m_pStringBuffer));

		//Indicate weither or not provider includes bookmark column
		m_fContainsBmk = FALSE;
		if(m_rgColumnInfo)
			m_fContainsBmk = m_rgColumnInfo[0].iOrdinal==0;
	}

CLEANUP:
	return hr;
}


/////////////////////////////////////////////////////////////////
// WCHAR* CRowset::GetColName
//
/////////////////////////////////////////////////////////////////
WCHAR* CRowset::GetColName(ULONG iOrdinal)
{
	HRESULT	hr = E_FAIL;

	static WCHAR* pwszNullName = L"<NULL>";
	static WCHAR* pwszEmptyName = L"<Empty>";
	static WCHAR* pwszBookmarkName = L"<Bookmark>";

	//Providers are not required to return a Column Name for any column
	//Useful information, but some columns (ie: Bookmarks) and others
	//May not have a column name.  So this method will either just return
	//Whatever the provider returned or will generate an appropiate one if not...
	//We really need a column name to help for the GUI headers

	DBCOLUMNINFO* pColInfo = GetColInfo(iOrdinal);
	if(pColInfo == NULL)
		return NULL;
	
	WCHAR* pwszColName = pColInfo->pwszName;
	
	if(pwszColName == NULL)
	{
		pwszColName = ((iOrdinal==0) ? pwszBookmarkName : pwszNullName);
	}
	else if(pwszColName[0] == wEOL)
	{
		pwszColName = ((iOrdinal==0) ? pwszBookmarkName : pwszEmptyName);
	}

	ASSERT(pwszColName);
	return pwszColName;
}


/////////////////////////////////////////////////////////////////
// DBCOLUMNINFO* CRowset::GetColInfo
//
/////////////////////////////////////////////////////////////////
DBCOLUMNINFO* CRowset::GetColInfo(ULONG iOrdinal)
{
	//Due to the bookmark column be present or absent causes the ColInfo
	//to not be lined exactly (1:1) to the Ordinal position...

	//no-ops
	if(m_cColumns==0 || m_rgColumnInfo == NULL || (iOrdinal==0 && m_rgColumnInfo[0].iOrdinal!=0))
		return NULL;
	
	if(m_rgColumnInfo[0].iOrdinal==0)
	{
		//[0,1,2,3,4,5...]
		ASSERT(iOrdinal < m_cColumns);
		ASSERT(m_rgColumnInfo[iOrdinal].iOrdinal == iOrdinal);
		return &m_rgColumnInfo[iOrdinal];
	}
	else
	{
		//[1,2,3,4,5...]
		ASSERT(iOrdinal && iOrdinal <= m_cColumns);
		ASSERT(m_rgColumnInfo[iOrdinal-1].iOrdinal == iOrdinal);
		return &m_rgColumnInfo[iOrdinal-1];
	}
	
	return NULL;
}


/////////////////////////////////////////////////////////////////
// DBBINDING* CRowset::GetBinding
//
/////////////////////////////////////////////////////////////////
DBBINDING* CRowset::GetBinding(ULONG iOrdinal)
{
	//Due to the bookmark column be present or absent causes the Binding
	//to not be lined exactly (1:1) to the Ordinal position...

	//no-ops
	if(m_cBindings==0 || m_rgBindings == NULL || (iOrdinal==0 && m_rgBindings[0].iOrdinal!=0))
		return NULL;
	
	//Bookmark
	if(m_rgBindings[0].iOrdinal == 0)
	{
		//[0,1,2,3,4,5...]
		ASSERT(iOrdinal < m_cColumns);
		ASSERT(m_rgBindings[iOrdinal].iOrdinal == iOrdinal);
		return &m_rgBindings[iOrdinal];
	}
	else
	{
		//[1,2,3,4,5...]
		ASSERT(iOrdinal && iOrdinal <= m_cColumns);
		ASSERT(m_rgBindings[iOrdinal-1].iOrdinal == iOrdinal);
		return &m_rgBindings[iOrdinal-1];
	}
	
	return NULL;
}


/////////////////////////////////////////////////////////////////
// HRESULT CRowset::ValidateRow
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::ValidateRow(HROW hRow, ULONG ulRefCount)
{
	//Warn user about NULL Rowhandles...
	if(hRow == DB_NULL_HROW || ulRefCount == 0)
	{
		if(IDNO == wMessageBox
			(
				NULL, 
				MB_TASKMODAL | MB_ICONQUESTION | MB_YESNO | MB_DEFBUTTON1, 
				wsz_ERROR, 
				L"hRow == 0x%08x\n"
				L"%s\n\n"
				L"This may crash your Provider...\n"
				L"Do you wish to continue anyway?", 
				hRow,
				hRow == DB_NULL_HROW ? L"Invalid Row Handle" : L"Released Row Handle"
				)
			)
			return E_FAIL;
	}

	return S_OK;
}

/////////////////////////////////////////////////////////////////
// HRESULT CRowset::ValidateAccessor
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::ValidateAccessor(HACCESSOR hAccessor, ULONG ulRefCount)
{
	//Warn user about NULL Accessor handle...
	if(hAccessor == DB_NULL_HACCESSOR || ulRefCount == 0)
	{
		if(IDNO == wMessageBox
			(
				NULL, 
				MB_TASKMODAL | MB_ICONQUESTION | MB_YESNO | MB_DEFBUTTON1, 
				wsz_ERROR, 
				L"hAccessor == 0x%08x\n"
				L"%s\n\n"
				L"This may crash your Provider...\n"
				L"Do you wish to continue anyway?", 
				hAccessor,
				hAccessor == DB_NULL_HROW ? L"Invalid Accessor Handle" : L"Released Accessor Handle"
			)
		  )
			return E_FAIL;
	}

	return S_OK;
}

/////////////////////////////////////////////////////////////////
// HRESULT CRowset::GetData
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::GetData(HROW hRow, HACCESSOR hAccessor, void* pData)
{
	//Since GetData is the only OLEDB method that is allowed to crash
	//if given invalid argumenets (for effiecientcy - provider speicific
	//weither it validates arguments), we should check the params and at the 
	//very least warn the user...
	ASSERT(m_pIRowset);
	HRESULT hr = E_FAIL;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	//Defaults
	if(hAccessor == NULL)
		hAccessor = m_hAccessor;
	if(pData == NULL)
		pData = m_pData;

	//Check Arguments...
	TESTC(hr = ValidateRow(hRow));
	TESTC(hr = ValidateAccessor(hAccessor));

	//GetData
 	if (m_eRowsetSource == ROWSET_FROMCOMMANDDATASET)
 	{
		hr = m_pCDataset->GetCellData(hRow, hAccessor, pData);
 	}
 	else 
 	{
 		TESTC(pCListBox->OutputPreMethod("IRowset:GetData(0x%08x, 0x%08x, 0x%08x)", hRow, hAccessor, pData));
		XTEST(hWnd, hr = m_pIRowset->GetData(hRow, hAccessor, pData));
		TESTC(pCListBox->OutputPostMethod(hr, "IRowset:GetData(0x%08x, 0x%08x, 0x%08x)", hRow, hAccessor, pData));
	}

CLEANUP:
	return hr;
}	
	

/////////////////////////////////////////////////////////////////
// HRESULT CRowset::GetOriginalData
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::GetOriginalData(HROW hRow, HACCESSOR hAccessor, void* pData)
{
	//Since GetData is the only OLEDB method that is allowed to crash
	//if given invalid argumenets (for effiecientcy - provider speicific
	//weither it validates arguments), we should check the params and at the 
	//very least warn the user...
	ASSERT(m_pIRowsetUpdate);
	HRESULT hr = S_OK;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	//Defaults
	if(hAccessor == NULL)
		hAccessor = m_hAccessor;
	if(pData == NULL)
		pData = m_pData;

	//Check Arguments...
	TESTC(hr = ValidateRow(hRow));
	TESTC(hr = ValidateAccessor(hAccessor));

	//GetOriginalData
	TESTC(pCListBox->OutputPreMethod("IRowsetUpdate:GetOriginalData(0x%08x, 0x%08x, 0x%08x)", hRow, hAccessor, pData));
	XTEST(hWnd, hr = m_pIRowsetUpdate->GetOriginalData(hRow, hAccessor, pData));
	TESTC(pCListBox->OutputPostMethod(hr, "IRowsetUpdate:GetOriginalData(0x%08x, 0x%08x, 0x%08x)", hRow, hAccessor, pData));

CLEANUP:
	return hr;
}	


/////////////////////////////////////////////////////////////////
// HRESULT CRowset::GetVisibleData
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::GetVisibleData(HROW hRow, HACCESSOR hAccessor, void* pData)
{
	//Since GetData is the only OLEDB method that is allowed to crash
	//if given invalid argumenets (for effiecientcy - provider speicific
	//weither it validates arguments), we should check the params and at the 
	//very least warn the user...
	ASSERT(m_pIRowsetResynch);
	HRESULT hr = S_OK;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	//Defaults
	if(hAccessor == NULL)
		hAccessor = m_hAccessor;
	if(pData == NULL)
		pData = m_pData;

	//Check Arguments...
	TESTC(hr = ValidateRow(hRow));
	TESTC(hr = ValidateAccessor(hAccessor));

	//GetVisibleData
	TESTC(pCListBox->OutputPreMethod("IRowsetResynch:GetVisibleData(0x%08x, 0x%08x, 0x%08x)", hRow, hAccessor, pData));
	XTEST(hWnd, hr = m_pIRowsetResynch->GetVisibleData(hRow, hAccessor, pData));
	TESTC(pCListBox->OutputPostMethod(hr, "IRowsetResynch:GetVisibleData(0x%08x, 0x%08x, 0x%08x)", hRow, hAccessor, pData));

CLEANUP:
	return hr;
}	


/////////////////////////////////////////////////////////////////
// HRESULT CRowset::GetBookmark
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::GetBookmark(HROW hRow, ULONG* pcbBookmark, BYTE** ppBookmark, void** ppData)
{
	ASSERT(pcbBookmark);
	ASSERT(ppBookmark);
	HRESULT hr = S_OK;
	*pcbBookmark = 0;
	*ppBookmark = NULL;
	
	DBSTATUS dwStatus = 0;
	DWORD dwLength = 0;
	
	if(m_cBmkBindings==0 || m_rgBmkBindings==NULL)
		return E_FAIL;

	//Allocate enough space for the bookmark
	//And Status Length of course!
	SAFE_ALLOC(*ppData, BYTE, m_rgBmkBindings[0].cbMaxLen + sizeof(DBSTATUS) + sizeof(ULONG) + 10);

	//GetData to obtain Bookmark value (delegate)
	TESTC(hr = GetData(hRow, m_hBmkAccessor, *ppData));
	
	//Output Data
	*pcbBookmark = BINDING_LENGTH(m_rgBmkBindings[0], *ppData);
	*ppBookmark  = (BYTE*)&BINDING_VALUE(m_rgBmkBindings[0], *ppData);

CLEANUP:
	return hr;
}

/////////////////////////////////////////////////////////////////
// HRESULT CRowset::GetNextRows
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::GetNextRows(LONG lOffset, LONG cRows, ULONG* pcRowsObtained, HROW** prghRows)
{
	ASSERT(m_pIRowset);
	HRESULT hr = S_OK;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	ULONG cRowsObtained = 0;
	HROW* rghRows = NULL;
	if(prghRows)
		rghRows = *prghRows;

	//GetNextRows
	TESTC(pCListBox->OutputPreMethod("IRowset::GetNextRows(NULL, %d, %d, &%d, &0x%08x)", lOffset, cRows, cRowsObtained, rghRows));
	XTEST(hWnd, hr = m_pIRowset->GetNextRows(
		NULL,						// hChapter
		lOffset,					// lOffset
		cRows,						// cRows
		&cRowsObtained,				// cRowsObtained
		&rghRows));					// rghRows
	TESTC(pCListBox->OutputPostMethod(hr, "IRowset::GetNextRows(NULL, %d, %d, &%d, &0x%08x)", lOffset, cRows, cRowsObtained, rghRows));

CLEANUP:
	if(pcRowsObtained)
		*pcRowsObtained = cRowsObtained;
	
	if(prghRows)
		*prghRows = rghRows;
	else
		SAFE_FREE(rghRows);
	return hr;
}	


/////////////////////////////////////////////////////////////////
// HRESULT CRowset::AddRefRows
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::AddRefRows(ULONG cRows, HROW* rghRows, ULONG* rgRefCounts)
{
	ASSERT(m_pIRowset);
	HRESULT hr = S_OK;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	//Alloc Array for Status
	DBROWSTATUS* rgRowStatus = NULL;
	SAFE_ALLOC(rgRowStatus, DBROWSTATUS, cRows);

	//AddRefRows
	TESTC(hr = pCListBox->OutputPreMethod("IRowset::AddRefRows(%d, 0x%08x, 0x%08x, 0x%08x)", cRows, rghRows, rgRefCounts, rgRowStatus));
	XTEST(hWnd, hr = m_pIRowset->AddRefRows(cRows, rghRows, rgRefCounts, rgRowStatus));
	if(hr==DB_S_ERRORSOCCURRED || hr==DB_E_ERRORSOCCURRED)
		DisplayRowErrors(hWnd, cRows, rghRows, rgRowStatus);
	TESTC(hr = pCListBox->OutputPostMethod(hr, "IRowset::AddRefRows(%d, 0x%08x, 0x%08x, 0x%08x)", cRows, rghRows, rgRefCounts, rgRowStatus));

CLEANUP:
	SAFE_FREE(rgRowStatus);
	return hr;
}	


/////////////////////////////////////////////////////////////////
// HRESULT CRowset::ReleaseRows
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::ReleaseRows(ULONG cRows, HROW* rghRows, ULONG* rgRefCounts)
{
	ASSERT(m_pIRowset);
	HRESULT hr = S_OK;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	//Alloc Array for Status
	DBROWSTATUS* rgRowStatus = NULL;
	SAFE_ALLOC(rgRowStatus, DBROWSTATUS, cRows);

	//ReleaseRows
	TESTC(hr = pCListBox->OutputPreMethod("IRowset::ReleaseRows(%d, 0x%08x, NULL, 0x%08x, 0x%08x)", cRows, rghRows, rgRefCounts, rgRowStatus));
	XTEST(hWnd, hr = m_pIRowset->ReleaseRows(cRows, rghRows, NULL, rgRefCounts, rgRowStatus));
	if(hr==DB_S_ERRORSOCCURRED || hr==DB_E_ERRORSOCCURRED)
		DisplayRowErrors(hWnd, cRows, rghRows, rgRowStatus);
	TESTC(hr = pCListBox->OutputPostMethod(hr, "IRowset::ReleaseRows(%d, 0x%08x, NULL, 0x%08x, 0x%08x)", cRows, rghRows, rgRefCounts, rgRowStatus));

CLEANUP:
	SAFE_FREE(rgRowStatus);
	return hr;
}	


////////////////////////////////////////////////////////////////
// CRowset::SetColumnData
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::SetColumnData(DBBINDING* pBinding, void* pData, DBSTATUS dwStatus, CHAR* pszValue)
{
	ASSERT(pBinding);
	ASSERT(pData);
	ASSERT(pszValue);

	WCHAR wszBuffer[MAX_COL_SIZE+1];
	VARIANT vVariant;
	VariantInit(&vVariant);
	HRESULT hr = S_OK;
	
	//set STATUS
	BINDING_STATUS(*pBinding, pData) = dwStatus;
	BINDING_LENGTH(*pBinding, pData) = 0;
	BINDING_VALUE(*pBinding, pData) = NULL;

	//STATUS
	switch(dwStatus)
	{
		case DBSTATUS_S_ISNULL:
		case DBSTATUS_S_DEFAULT:
			break;

		default:	//DBSTATUS_S_OK
		{
			//Set STATUS
			BINDING_STATUS(*pBinding, pData) = DBSTATUS_S_OK;
			
			//We have a problem if the user has entered a value (string) longer than
			//the buffer we have allocated.  If the length is greater than
			//cbMaxLen we need to truncate it and inform the user...
			ULONG cbBytes = strlen(pszValue);
			switch(pBinding->wType)
			{
				case DBTYPE_STR:
					cbBytes = (cbBytes + 1)*sizeof(CHAR);
					break;
				
				case DBTYPE_WSTR:
				case DBTYPE_BSTR:
					cbBytes = (cbBytes + 1)*sizeof(WCHAR);
					break;

				case DBTYPE_BYTES:
					cbBytes = cbBytes >> 1;
					break;
			};

			if(cbBytes > pBinding->cbMaxLen && 
				pBinding->wType != DBTYPE_VARIANT)
			{
				//Truncate the Value 
				if(pBinding->wType == DBTYPE_WSTR || pBinding->wType == DBTYPE_BSTR)
					pszValue[pBinding->cbMaxLen/2-1] = EOL;
				else if (pBinding->wType == DBTYPE_STR)
					pszValue[pBinding->cbMaxLen-1] = EOL;
				else if (pBinding->wType == DBTYPE_BYTES)
					pszValue[pBinding->cbMaxLen<<1] = EOL;

				//Inform user of Truncation...
				wMessageBox(
					m_pCMDIChild->m_hWnd, MB_TASKMODAL | MB_ICONHAND | MB_OK | MB_DEFBUTTON1, 
					wsz_ERROR, 
					L"You have entered a value for column %d of %d bytes\n\n"
					L"The cbMaxSize of this column is %d.\n"
					L"Your value will be truncated to\n"
					L"(%S)",
					pBinding->iOrdinal, 
					cbBytes, 
					pBinding->cbMaxLen, 
					pszValue 
					);			
			}

			//Set LENGTH and VALUE
			//We may have bound this column in the accessor differently...
			switch(pBinding->wType)
			{
				case DBTYPE_WSTR:
					ConvertToWCHAR(pszValue, wszBuffer, MAX_COL_SIZE);
					BINDING_LENGTH(*pBinding, pData) = cbBytes-sizeof(WCHAR);
					wcscpy((WCHAR*)&BINDING_VALUE(*pBinding, pData), wszBuffer);
					break;

				case DBTYPE_STR:
					BINDING_LENGTH(*pBinding, pData) = cbBytes-sizeof(CHAR);
					strcpy((CHAR*)&BINDING_VALUE(*pBinding, pData), pszValue);
					break;
					
				case DBTYPE_VARIANT:						
					ConvertToWCHAR(pszValue, wszBuffer, MAX_COL_SIZE);
					V_VT(&vVariant) = VT_BSTR;
					V_BSTR(&vVariant) = SysAllocString(wszBuffer);
					BINDING_LENGTH(*pBinding, pData) = sizeof(VARIANT);
					memcpy(&BINDING_VALUE(*pBinding, pData), &vVariant, sizeof(VARIANT));
					break;

				case DBTYPE_BYTES:
					XTESTC(m_pCMDIChild->m_hWnd, hr = StringToBYTES(pszValue, (BYTE*)&BINDING_VALUE(*pBinding, pData), cbBytes, &BINDING_LENGTH(*pBinding, pData)));
					break;

				case DBTYPE_IUNKNOWN: 
					//TODO
					//This is kind of a pain for SetData.  Since I'm going to
					//have to implement each object in order to pass it to the 
					//provider.  We might also be able to try the STGM_CREATE
					//which is supposed to return an empty implemented object
					//that we could "pump" our data into...
					
					//For now, the value is just set to NULL
					break;

				default:
					ASSERT(!"Unhandled Type!");
					break;
			}
			break;
		}
	}

CLEANUP:
	return hr;
}


////////////////////////////////////////////////////////////////
// CRowset::GetColumnData
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::GetColumnData(DBBINDING* pBinding, void* pData, DBSTATUS* pdwStatus, ULONG* pdwLength, DBTYPE* pwSubType, CHAR* pszValue, ULONG ulMaxSize, DWORD dwFlags, DBTYPE wBaseType)
{
	ASSERT(pBinding);
	ASSERT(pData);
	ASSERT(pszValue);
	HRESULT hr = S_OK;

	//STATUS
	DBSTATUS dwStatus = BINDING_STATUS(*pBinding, pData);

	//LENGTH
	DWORD dwLength = BINDING_LENGTH(*pBinding, pData);
	
	//SUB-DBTYPE
	DBTYPE	wSubType = wBaseType == DBTYPE_VARIANT ? DBTYPE_EMPTY : wBaseType;
	
	//LENGTH is the Actual (Untruncated) length of the string
	//We we need to make sure all operations are off of cbMaxLen and not LENGTH bound
	ULONG dwActualLength = min(pBinding->cbMaxLen-1, dwLength);
	ASSERT(pBinding->cbMaxLen <= ulMaxSize);

	//VALUE
	pszValue[0] = EOL;
	switch(dwStatus)
	{
		case DBSTATUS_S_ISNULL:
			break;

		//Let this case "fall" through to the S_OK case...
		case DBSTATUS_S_TRUNCATED:
			hr = DB_S_ERRORSOCCURRED;

		case DBSTATUS_S_OK:
		case DBSTATUS_S_DEFAULT:
		{
			switch(pBinding->wType)
			{
				case DBTYPE_WSTR:
				{	
					WCHAR* pwszData = (WCHAR*)&BINDING_VALUE(*pBinding, pData);
					ConvertToMBCS(pwszData, pszValue, ulMaxSize);
					break;
				}

				case DBTYPE_STR:
				{	
					CHAR* pszData = (CHAR*)&BINDING_VALUE(*pBinding, pData);
					strncpy(pszValue, pszData, dwActualLength);
					pszValue[dwActualLength] = EOL;
					break;
				}
					
				case DBTYPE_BYTES:
				{	
					//Otherwise treat it as a stream of BYTES
					BYTE* pbData = (BYTE*)&BINDING_VALUE(*pBinding, pData);
					BYTESToString(pbData, pszValue, dwActualLength, &dwLength);
					break;
				}

				case DBTYPE_VARIANT:
				{
					VARIANT* pVariant = (VARIANT*)&BINDING_VALUE(*pBinding, pData);
					if(pVariant)
					{
						wSubType = V_VT(pVariant);
						hr = VariantToString(pVariant, pszValue, ulMaxSize, dwFlags);
					}
					break;
				}

				case DBTYPE_IUNKNOWN: 
				{
					//This is required for ColumnsRowset IUnknown column
					//IUnknown is not a required conversion to string (WSTR)
					IUnknown* pIUnknown = (IUnknown*)BINDING_VALUE(*pBinding, pData);
					REFIID iid = pBinding->pObject ? pBinding->pObject->iid : IID_NULL;
					
					WCHAR wszBuffer[MAX_QUERY_LEN] = {0};
					void* pvBuffer = pszValue;
					dwLength = 0;

					//Need to handle WIDE streams...
					if(wBaseType == DBTYPE_BYTES || wBaseType == DBTYPE_WSTR || wBaseType == DBTYPE_BSTR)
						pvBuffer = wszBuffer;

					if(pIUnknown == NULL)
					{
						//I have no clue what type of object this is.
						//So just display the pointer
						sprintf(pszValue, "0x%08x", pIUnknown);
					}
					else if(iid == IID_ISequentialStream)
					{
						//Read the Data into our buffer
						ISequentialStream* pISeqStream = (ISequentialStream*)pIUnknown;
						XTESTC(NULL, hr = pISeqStream->Read(pvBuffer, ulMaxSize, &dwLength));
					}
					else if(iid == IID_ILockBytes)
					{
						//Read the Data into our buffer
						ULARGE_INTEGER ulOffset = {0};
						ILockBytes* pILockBytes = (ILockBytes*)pIUnknown;
						XTESTC(NULL, hr = pILockBytes->ReadAt(ulOffset, pvBuffer, ulMaxSize, &dwLength));
					}
					else if(iid == IID_IStream)
					{
						//Read the Data into our buffer
						IStream* pIStream = (IStream*)pIUnknown;
						XTESTC(NULL, hr = pIStream->Read(pvBuffer, ulMaxSize, &dwLength));
					}
//TODO				else if(iid == IID_IStorage)
//					{
//					}
					else
					{
						//I have no clue what type of object this is.
						//So just display the pointer
						sprintf(pszValue, "0x%08x", pIUnknown);
					}

					//Deal with the data returned...
					if(dwLength)
					{
						switch(wBaseType)
						{
							case DBTYPE_BSTR:
							case DBTYPE_WSTR:
								//Need to convert the stream to CHAR for display
								ConvertToMBCS(wszBuffer, pszValue, min(dwLength, ulMaxSize-1));
								break;

							case DBTYPE_STR:
								break;

							default:
								//Otherwise treat it as a stream of BYTES
								BYTESToString((BYTE*)pvBuffer, pszValue, min(dwLength, ulMaxSize-1), &dwLength);
								break;
						}

						//Need to NULL terminate the stream, since its just a stream
						//of bytes returned from the provider with no termination
						pszValue[min(dwLength, ulMaxSize-1)] = EOL;
					}

					SAFE_RELEASE(pIUnknown);
					break;
				}

				default:
					ASSERT(!"Unhandled Type!");
					hr = E_FAIL;
					break;
			}
			break;
		}

		default:
			//Error Status code...
			hr = E_FAIL;
			break;
	}
	
	//PostProcessing 
	if(SUCCEEDED(hr))
	{
		//Hexadecimal
		if(dwFlags & CONV_HEX)
		{
			switch(wSubType)
			{
				case DBTYPE_I1:
				case DBTYPE_UI1:
				case DBTYPE_I2:
				case DBTYPE_UI2:
				case DBTYPE_I4:
				case DBTYPE_UI4:
					ConvertString(pszValue, ulMaxSize, dwFlags);
					break;
			}
		}
	}


CLEANUP:
	//Output Params
	if(pdwStatus)
		*pdwStatus = dwStatus;
	if(pdwLength)
		*pdwLength = dwLength;
	if(pwSubType)
		*pwSubType = wSubType;
	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CRowset::CreateAccessors
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::CreateAccessors(BOOL fBindBookmark)
{
	HRESULT hr = S_OK;
	HRESULT hrOptional = S_OK;
	ULONG cbRowSize = 0;
	
	//Release Previous Accessors
	ReleaseAccessor(&m_hAccessor);
	FreeBindings(&m_cBindings, &m_rgBindings);
	SAFE_FREE(m_pData);

	ReleaseAccessor(&m_hBmkAccessor);
	FreeBindings(&m_cBmkBindings, &m_rgBmkBindings);

	//Delegate
	TESTC(hr = SetupBindings(fBindBookmark ? BIND_ALLCOLS : BIND_ALLCOLSEXPECTBOOKMARK, &m_cBindings, &m_rgBindings, &cbRowSize));
	TESTC(hr = CreateAccessor(DBACCESSOR_ROWDATA, m_cBindings, m_rgBindings, cbRowSize, &m_hAccessor));

	//Allocate pData
	SAFE_ALLOC(m_pData, BYTE, cbRowSize);

	//Create Bookmark Accessor (if bookmark is included)
	if(m_fContainsBmk)
	{
		TESTC(hrOptional = SetupBindings(BIND_BOOKMARKONLY, &m_cBmkBindings, &m_rgBmkBindings, &cbRowSize));
		TESTC(hrOptional = CreateAccessor(DBACCESSOR_ROWDATA, m_cBmkBindings, m_rgBmkBindings, cbRowSize, &m_hBmkAccessor));
	}

CLEANUP:
	return hr;
}
	

/////////////////////////////////////////////////////////////////
// HRESULT CRowset::SetupBindings
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::SetupBindings(BINDCOLS eBindCols, ULONG* pcBindings, DBBINDING** prgBindings, ULONG* pcRowSize, BOOL* pbOutofLine)
{
	HRESULT hr = S_OK;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	ULONG dwOffset = 0;
	ULONG i,cBindings = 0;
	DBBINDING* rgBindings = NULL;
	ULONG cStorageObjects = 0;
	
	//Only capable of the Following Converions (for Display)
	DBTYPE wBindingType = (DBTYPE)GetOptionsObj()->m_dwBindingType;
	DWORD  dwRowsetOpts = GetOptionsObj()->m_dwRowsetOpts;
	
	//Special Handling for Bookmark only accessor
	if(eBindCols == BIND_BOOKMARKONLY)
		wBindingType = DBTYPE_BYTES;
	else if(m_eRowsetSource == ROWSET_FROMCOMMANDDATASET)
 		wBindingType = DBTYPE_VARIANT;	// MSOLAP supports only variants

	//Alloc the space to hold the Bindings
	SAFE_ALLOC(rgBindings, DBBINDING, m_cColumns);

	cBindings = 0;
	for(i=0; i<m_cColumns; i++) 
	{
		ASSERT(m_rgColumnInfo);

		//Setup the Bindings
		rgBindings[cBindings].iOrdinal	= m_rgColumnInfo[i].iOrdinal;
		rgBindings[cBindings].obStatus	= dwOffset;
		rgBindings[cBindings].obLength	= dwOffset + sizeof(DBSTATUS);
		rgBindings[cBindings].obValue	= dwOffset + sizeof(DBSTATUS) + sizeof(ULONG);
		
		rgBindings[cBindings].pTypeInfo = NULL;
		rgBindings[cBindings].pBindExt  = NULL;

		rgBindings[cBindings].dwPart	= DBPART_VALUE|DBPART_LENGTH|DBPART_STATUS;			
		rgBindings[cBindings].dwMemOwner= DBMEMOWNER_CLIENTOWNED;
		rgBindings[cBindings].eParamIO	= DBPARAMIO_NOTPARAM;
		
		rgBindings[cBindings].dwFlags	= 0;
		rgBindings[cBindings].bPrecision= m_rgColumnInfo[i].bPrecision;
		rgBindings[cBindings].bScale	= m_rgColumnInfo[i].bScale;

		rgBindings[cBindings].pObject	= NULL;
		rgBindings[cBindings].wType		= wBindingType;
		rgBindings[cBindings].cbMaxLen	= 0;
						
		//May need to adjust the MaxLen, depending upon what the BindingType is
		switch(wBindingType)
		{
			case DBTYPE_VARIANT:
				rgBindings[cBindings].cbMaxLen = sizeof(VARIANT);
				break;

			case DBTYPE_BYTES:
				rgBindings[cBindings].cbMaxLen	= min(m_rgColumnInfo[i].ulColumnSize, MAX_COL_SIZE);
				break;

			//Strings are kind of a pain.  Although we get the luxury of 
			//Having the provider coerce the type, we need to allocate a buffer 
			//large enough for the provider to store the type in string format
			case DBTYPE_STR:
			case DBTYPE_WSTR:
				//Account for TYPE -> String
				//(make sure enough room in buffer)
				switch(m_rgColumnInfo[i].wType)
				{
					case DBTYPE_NULL:
					case DBTYPE_EMPTY:
						//Don't need much room for these...
						break;

					case DBTYPE_I1:
					case DBTYPE_I2:
					case DBTYPE_I4:
					case DBTYPE_UI1:
					case DBTYPE_UI2:
					case DBTYPE_UI4:
					case DBTYPE_R4:
						//All of the above fit well into 15 characters of display size
						rgBindings[cBindings].cbMaxLen = 15;
						break;

					case DBTYPE_I8:
					case DBTYPE_UI8:
					case DBTYPE_R8:
					case DBTYPE_CY:
					case DBTYPE_ERROR:
					case DBTYPE_BOOL:
						//All of the above fit well into 25 characters of display size
						rgBindings[cBindings].cbMaxLen = 25;
						break;

					case DBTYPE_DECIMAL:
					case DBTYPE_NUMERIC:
					case DBTYPE_DATE:
					case DBTYPE_DBDATE:
					case DBTYPE_DBTIMESTAMP:
					case DBTYPE_GUID:
						//All of the above fit well into 50 characters of display size
						rgBindings[cBindings].cbMaxLen = 50;
						break;
					
					case DBTYPE_BYTES:
						//Bytes -> String, 1 byte = 2 Ascii chars. (0xFF == "FF")
						rgBindings[cBindings].cbMaxLen	= min(m_rgColumnInfo[i].ulColumnSize, MAX_COL_SIZE) * 2;
						break;

					case DBTYPE_STR:
					case DBTYPE_WSTR:
					case DBTYPE_BSTR:
						//String -> String
						//cbMaxLen already contains the length, and below we will 
						//account for the NULL terminator as well...
						rgBindings[cBindings].cbMaxLen	= min(m_rgColumnInfo[i].ulColumnSize, MAX_COL_SIZE);
						break;

					default:
 						//For everything else, VARIANT, IUNKNOWN, UDT, etc
						//Just default to our largest buffer size
						rgBindings[cBindings].cbMaxLen	= MAX_COL_SIZE;
						break;
				};
				break;

			
			//Currently we only handle binding to STR, WSTR, VARIANT
			default:
				ASSERT(!"Unhandled Binding Type!");
				break;
		};
					
		//BLOB or IUnknown Bindings
		if(m_rgColumnInfo[i].wType == DBTYPE_IUNKNOWN || 
			(m_rgColumnInfo[i].dwFlags & DBCOLUMNFLAGS_ISLONG && (dwRowsetOpts & (ROWSET_BLOB_ISEQSTREAM|ROWSET_BLOB_ILOCKBYTES|ROWSET_BLOB_ISTORAGE|ROWSET_BLOB_ISTREAM))))
		{
			//For ColumnsRowset
			rgBindings[cBindings].cbMaxLen	= MAX_COL_SIZE;
			rgBindings[cBindings].wType = DBTYPE_IUNKNOWN;

			//Create pObject
			SAFE_ALLOC(rgBindings[cBindings].pObject, DBOBJECT, 1);
			rgBindings[cBindings].pObject->dwFlags	= STGM_READ;
			rgBindings[cBindings].pObject->iid		= IID_ISequentialStream;

			//Setup pObject->iid 
			if(dwRowsetOpts & ROWSET_BLOB_ISTORAGE)
				rgBindings[cBindings].pObject->iid		= IID_IStorage;
			else if(dwRowsetOpts & ROWSET_BLOB_ISTREAM)
				rgBindings[cBindings].pObject->iid		= IID_IStream;
			else if(dwRowsetOpts & ROWSET_BLOB_ILOCKBYTES)
				rgBindings[cBindings].pObject->iid		= IID_ILockBytes;
			else
				rgBindings[cBindings].pObject->iid		= IID_ISequentialStream;
		}	

		//ulColumnSize - is count of Chars for Strings		
		if(rgBindings[cBindings].wType == DBTYPE_WSTR)
			rgBindings[cBindings].cbMaxLen *= 2;

		//Account for the NULL terminator
		if(rgBindings[cBindings].wType == DBTYPE_STR)
			rgBindings[cBindings].cbMaxLen	+= sizeof(CHAR);
		if(rgBindings[cBindings].wType == DBTYPE_WSTR)
			rgBindings[cBindings].cbMaxLen	+= sizeof(WCHAR);

		//MAX_LENGTH
		rgBindings[cBindings].cbMaxLen	= min(rgBindings[cBindings].cbMaxLen, MAX_COL_SIZE);
		dwOffset = rgBindings[cBindings].cbMaxLen + rgBindings[cBindings].obValue;
		dwOffset = ROUNDUP( dwOffset );

		switch(eBindCols)
		{
			case BIND_ALLCOLS:
				cBindings++;
				break;

			case BIND_ALLCOLSEXPECTBOOKMARK:
				if(rgBindings[cBindings].iOrdinal != 0)
					cBindings++;
				break;

			case BIND_UPDATEABLECOLS:
				if(m_rgColumnInfo[i].dwFlags & DBCOLUMNFLAGS_WRITE || m_rgColumnInfo[i].dwFlags & DBCOLUMNFLAGS_WRITEUNKNOWN)
					cBindings++;
				break;

			case BIND_BOOKMARKONLY:
				rgBindings[cBindings].iOrdinal = 0;
				cBindings++;
				i = m_cColumns;
				break;

			default:
				ASSERT(!"Unhandled Type!");
				break;
		}
	}

	//Size for pData
	if(pcRowSize)
		*pcRowSize = dwOffset;

CLEANUP:
	//Accessors
	if(pcBindings)
		*pcBindings = cBindings;
	if(prgBindings)
		*prgBindings = rgBindings;
	else
		SAFE_FREE(rgBindings);
	
	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CRowset::CreateAccessor
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::CreateAccessor(DBACCESSORFLAGS dwAccessorFlags, ULONG cBindings, DBBINDING* rgBindings, ULONG cRowSize, HACCESSOR* phAccessor)
{
	ASSERT(phAccessor);
	ASSERT(m_pIAccessor);

	HRESULT hr = S_OK;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	DBBINDSTATUS* rgStatus = NULL;

	//Alloc the space to hold the status
	SAFE_ALLOC(rgStatus, DBBINDSTATUS, cBindings);

	//Create the accessor
	*phAccessor = NULL;
	TESTC(pCListBox->OutputPreMethod("IAccessor::CreateAccessor(%d, %d, 0x%08x, %d, &0x%08x, 0x%08x)", dwAccessorFlags, cBindings, rgBindings, cRowSize, *phAccessor, rgStatus));
	XTEST(hWnd, hr = m_pIAccessor->CreateAccessor(dwAccessorFlags, cBindings, rgBindings, cRowSize, phAccessor, rgStatus));
	if(hr == DB_S_ERRORSOCCURRED || hr == DB_E_ERRORSOCCURRED)
		DisplayAccessorErrors(NULL, cBindings, rgBindings, rgStatus);
	TESTC(pCListBox->OutputPostMethod(hr, "IAccessor::CreateAccessor(%d, %d, 0x%08x, %d, &0x%08x, 0x%08x)", dwAccessorFlags, cBindings, rgBindings, cRowSize, *phAccessor, rgStatus));

CLEANUP:
	SAFE_FREE(rgStatus);
	return hr;
}
	
/////////////////////////////////////////////////////////////////
// HRESULT CRowset::ReleaseAccessor
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::ReleaseAccessor(HACCESSOR* phAccessor)
{
	ASSERT(phAccessor);
	HRESULT hr = S_OK;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	ULONG ulRefCount = 0;

	if(m_pIAccessor && *phAccessor)
	{
		TESTC(hr = pCListBox->OutputPreMethod("IAccessor::ReleaseAccessor(0x%08x, &%d)", *phAccessor, ulRefCount));
		XTEST(hWnd, hr = m_pIAccessor->ReleaseAccessor(*phAccessor, &ulRefCount));
		TESTC(hr = pCListBox->OutputPostMethod(hr, "IAccessor::ReleaseAccessor(0x%08x, &%d)", *phAccessor, ulRefCount));
	}

CLEANUP:
	//Only NULL the handle if the RefCount is 0
	if(ulRefCount == 0)
		*phAccessor = NULL;
	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CRowset::CreateRowset
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::CreateRowset(ROWSETSOURCE eRowsetSource, ULONG cPropSets, DBPROPSET* rgPropSets, LONG* pcRowsAffected, DBID* pTableID, DBID* pIndexID, REFIID riid, const GUID* pGuidSchema, ULONG cRestrictions, VARIANT* rgRestrictions)
{
	HRESULT				hr = S_OK;				// HRESULT
	HRESULT				hrOptional = S_OK;		// HRESULT

	ULONG				cOptColumns = 0;
	DBID*				rgOptColumns = NULL;
	IUnknown*			pIUnknown = NULL;
	IMultipleResults*	pIMultipleResults = NULL;

	ULONG  ulValue = 0;
	BOOL   bValue = FALSE;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	COptionsDlg* pCOptionsDlg = GetOptionsObj();

	if(pcRowsAffected)
		*pcRowsAffected = DB_COUNTUNAVAILABLE;

	//Schema Rowset
	if(eRowsetSource == ROWSET_FROMSCHEMA)
	{
		WCHAR wszBuffer[MAX_NAME_LEN+1];
		CHAR szBuffer[MAX_NAME_LEN+1];
		GUID guidSchema = pGuidSchema ? *pGuidSchema : GUID_NULL;
		CHAR* pszSchemaName = GetSchemaName(guidSchema);

		//Try to find the String Resprentation of the guidSchema
		if(pszSchemaName == NULL)
		{
			StringFromGUID2(guidSchema, wszBuffer, MAX_NAME_LEN);
			ConvertToMBCS(wszBuffer, szBuffer, MAX_NAME_LEN);
			pszSchemaName = szBuffer;
		}
		
		//GetSchema Rowset
		ASSERT(m_pCSession->m_pIDBSchemaRowset);
		TESTC(pCListBox->OutputPreMethod("IDBSchemaRowset::GetRowset(NULL, %s, %d, 0x%08x, %s, %d, 0x%08x, &0x%08x)", pszSchemaName, cRestrictions, rgRestrictions, GetInterfaceName(riid), cPropSets, rgPropSets, pIUnknown));
		XTEST_(hWnd, hr = m_pCSession->m_pIDBSchemaRowset->GetRowset(
							NULL,				// punkOuter
							guidSchema,			// schema IID
							cRestrictions,		// # of restrictions
							rgRestrictions,		// array of restrictions
							riid,				// rowset interface
							cPropSets,			// # of properties
							rgPropSets,			// properties
							&pIUnknown),S_OK);	// rowset pointer
		
		TESTC(pCListBox->OutputPostMethod(hr, "IDBSchemaRowset::GetRowset(NULL, %s, %d, 0x%08x, %s, %d, 0x%08x, &0x%08x)", pszSchemaName, cRestrictions, rgRestrictions, GetInterfaceName(riid), cPropSets, rgPropSets, pIUnknown));
	}
	//Rowset from ColumnsRowset
	else if(eRowsetSource == ROWSET_FROMCOMMANDCOLUMNSROWSET || eRowsetSource == ROWSET_FROMROWSETCOLUMNSROWSET)
	{
		//Which ColumnsRowset interface, Command or Rowset?
		IColumnsRowset* pIColumnsRowset = (eRowsetSource == ROWSET_FROMCOMMANDCOLUMNSROWSET ? m_pCCommand->m_pIColumnsRowset : m_pIColumnsRowset);
		ASSERT(pIColumnsRowset);

		//GetAvailableColumns
		//We just use cRestrictions passed in as the flag
		//to indicate weither to use Optional Columns
		if(cRestrictions)
		{
			TESTC(pCListBox->OutputPreMethod("IColumnsRowset::GetAvailableColumns(&%d, &0x%08x)", cOptColumns, rgOptColumns));
			XTEST(hWnd, hr = pIColumnsRowset->GetAvailableColumns(&cOptColumns, &rgOptColumns));
			TESTC(pCListBox->OutputPostMethod(hr, "IColumnsRowset::GetAvailableColumns(&%d, &0x%08x)", cOptColumns, rgOptColumns));
		}

		//GetColumnsRowset
		TESTC(pCListBox->OutputPreMethod("IColumnsRowset::GetColumnsRowset(NULL, %d, 0x%08x, %s, %d, 0x%08x, &0x%08x)", cOptColumns, rgOptColumns, GetInterfaceName(riid), cPropSets, rgPropSets, pIUnknown));
		XTEST_(hWnd, hr = pIColumnsRowset->GetColumnsRowset(
							NULL,				// punkOuter
							cOptColumns,		// cOptionalColumns
							rgOptColumns,		// rgOptionalColumns
							riid,				// rowset interface
							cPropSets,			// # of properties
							rgPropSets,			// properties
							&pIUnknown),S_OK);	// rowset pointer
		TESTC(pCListBox->OutputPostMethod(hr, "IColumnsRowset::GetColumnsRowset(NULL, %d, 0x%08x, %s, %d, 0x%08x, &0x%08x)", cOptColumns, rgOptColumns, GetInterfaceName(riid), cPropSets, rgPropSets, pIUnknown));
	}
	//Rowset from Enumerator
	else if(eRowsetSource == ROWSET_FROMENUM)
	{
		//GetSourcesRowset
		ASSERT(m_pCDataSource->m_pISourcesRowset);
		TESTC(pCListBox->OutputPreMethod("ISourcesRowset::GetSourcesRowset(NULL, %s, %d, 0x%08x, &0x%08x)", GetInterfaceName(riid), cPropSets, rgPropSets, pIUnknown));
		XTEST_(hWnd, hr = m_pCDataSource->m_pISourcesRowset->GetSourcesRowset(
							NULL,				// punkOuter
							riid,			// rowset interface
							cPropSets,			// # of properties
							rgPropSets,			// properties
							&pIUnknown),S_OK);	// rowset pointer
		TESTC(pCListBox->OutputPostMethod(hr, "ISourcesRowset::GetSourcesRowset(NULL, %s, %d, 0x%08x, &0x%08x)", GetInterfaceName(riid), cPropSets, rgPropSets, pIUnknown));
	}
	//Rowset from Command
	else if(eRowsetSource == ROWSET_FROMCOMMAND || eRowsetSource == ROWSET_FROMCOMMANDWITHPARAMETERS) 
	{
		DBPARAMS* pDispParams = NULL;
		if(eRowsetSource == ROWSET_FROMCOMMANDWITHPARAMETERS)
			pDispParams = &m_pCCommand->m_ParamInfo;
		
		//Clear previous rowset (if requested)
		if(pCOptionsDlg->m_dwCommandOpts & COMMAND_RELEASEROWSET)
			ReleaseRowset();

		//SetCommandText (unless requested not to...)
		if(!(pCOptionsDlg->m_dwCommandOpts & COMMAND_NOCOMMANDTEXT))
		{
			ASSERT(pTableID);
			TESTC(hr = m_pCCommand->SetCommandText(pTableID->uName.pwszName));
		}

		//Execute
		TESTC(pCListBox->OutputPreMethod("ICommand::Execute(NULL, %s, NULL, &%d, &0x%08x)", GetInterfaceName(riid), pcRowsAffected ? *pcRowsAffected : 0, pIUnknown));
		XTEST_(hWnd, hr = m_pCCommand->m_pICommand->Execute(
							NULL,					// pUnkOuter
							riid,					// refiid
							pDispParams,			// disp parms
							pcRowsAffected,			// rows affected
 							&pIUnknown),S_OK);  	// IRowset pointer
		TESTC(pCListBox->OutputPostMethod(hr, "ICommand::Execute(NULL, %s, NULL, &%d, &0x%08x)", GetInterfaceName(riid), pcRowsAffected ? *pcRowsAffected : 0, pIUnknown));
	}
	//IOpenRowset
	else if(eRowsetSource == ROWSET_FROMOPENROWSET)
	{
		ASSERT(m_pCSession->m_pIOpenRowset);
		WCHAR wszTableName[MAX_NAME_LEN+1];
		WCHAR wszIndexName[MAX_NAME_LEN+1];

		// From IOpenRowset, get a rowset object
		DBIDToString(pTableID, wszTableName, MAX_NAME_LEN);
		DBIDToString(pIndexID, wszIndexName, MAX_NAME_LEN);
		TESTC(pCListBox->OutputPreMethod("IOpenRowset::OpenRowset(NULL, %S, %S, %s, %d, 0x%08x, &0x%08x)", wszTableName, wszIndexName, GetInterfaceName(riid), cPropSets, rgPropSets, pIUnknown));
		XTEST_(hWnd, hr = m_pCSession->m_pIOpenRowset->OpenRowset(
								NULL,				// pUnkOuter
								pTableID,			// pTableID
								pIndexID,			// pIndexID
								riid,				// refiid
								cPropSets,			// cProperties
								rgPropSets,			// rgProperties
								&pIUnknown),S_OK);	// IRowset pointer
		TESTC(pCListBox->OutputPostMethod(hr, "IOpenRowset::OpenRowset(NULL, %S, %S, %s, %d, 0x%08x, &0x%08x)", wszTableName, wszIndexName, GetInterfaceName(riid), cPropSets, rgPropSets, pIUnknown));
	}
  	// Override the IRowset interface ptr with IMDDataset
  	else if(eRowsetSource == ROWSET_FROMCOMMANDDATASET)
  	{
  		// Clear previous rowset
  		ReleaseRowset();
  		m_pCDataset->RestartPosition();
  
  		TESTC(hr = m_pCDataset->GetIMDDataset(riid, &pIUnknown));
 	}
  	// Use the axis rowset to obtain IRowset interface ptr
  	else if (eRowsetSource == ROWSET_FROMAXISDATASET)
  	{
  		// Clear previous rowset
  		ReleaseRowset();
  		XTESTC(hWnd, hr = m_pCDataset->GetAxisRowset(riid, &pIUnknown));
  	}
	else
	{
		ASSERT(!"Unhandled Source");
		hr = E_FAIL;
		goto CLEANUP;
	}
	

	//Special Handling for IMultipleResults
	if(pIUnknown && (riid == IID_IMultipleResults || SUCCEEDED(pCListBox->OutputQI(pIUnknown, IID_IMultipleResults, (IUnknown**)&pIMultipleResults))))
	{
		//Obtain IMultipleResults
		TESTC(hr = m_pCMultipleResults->CreateObject(pIUnknown));
		pCListBox->OutputRelease(&pIUnknown);
		
		//Get the First Rowset...
		TESTC(hr = m_pCMultipleResults->GetResult(IID_IUnknown, pcRowsAffected, &pIUnknown));
	}

	//Display Errors (if occurred)
	if(hr == DB_S_ERRORSOCCURRED || hr == DB_E_ERRORSOCCURRED)
	{
		switch(eRowsetSource)
		{
			case ROWSET_FROMCOMMANDWITHPARAMETERS:
				DisplayPropErrors(hWnd, COMMAND, m_pCCommand->m_pICommand);
				DisplayBindingErrors(hWnd, m_pCCommand->m_cParamBindings, m_pCCommand->m_rgParamBindings, m_pCCommand->m_ParamInfo.pData);
				break;

			case ROWSET_FROMCOMMAND:
				DisplayPropErrors(hWnd, COMMAND, m_pCCommand->m_pICommand);
				break;

			default:
				DisplayPropErrors(hWnd, cPropSets, rgPropSets);
				break;
		};
	}

	if(!pIUnknown)
	{
		hr = S_FALSE;
		goto CLEANUP;
	}

	// Save rowset source
 	m_eRowsetSource = eRowsetSource;
		
	//Delegate - now that we have a rowset pointer...
	TESTC(hr = CreateRowset(pIUnknown));

CLEANUP:
	//Display Errors (if occurred)
	if(hr == DB_S_ERRORSOCCURRED || hr == DB_E_ERRORSOCCURRED)
	{
		switch(eRowsetSource)
		{
			case ROWSET_FROMCOMMANDWITHPARAMETERS:
				DisplayPropErrors(hWnd, COMMAND, m_pCCommand->m_pICommand);
				DisplayBindingErrors(hWnd, m_pCCommand->m_cParamBindings, m_pCCommand->m_rgParamBindings, m_pCCommand->m_ParamInfo.pData);
				break;

			case ROWSET_FROMCOMMAND:
				DisplayPropErrors(hWnd, COMMAND, m_pCCommand->m_pICommand);
				break;

			default:
				DisplayPropErrors(hWnd, cPropSets, rgPropSets);
				break;
		};
	}
	
	pCListBox->OutputRelease(&pIUnknown);
	pCListBox->OutputRelease((IUnknown**)&pIMultipleResults,  "IMultipleResults");
	SAFE_FREE(rgOptColumns);
	return hr;    
}



/////////////////////////////////////////////////////////////////
// HRESULT CRowset::CreateRowset
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::CreateRowset(IUnknown* pIUnknown)
{
	HRESULT				hr = S_OK;				// HRESULT
	HRESULT				hrOptional = S_OK;		// HRESULT

	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	COptionsDlg* pCOptionsDlg = GetOptionsObj();

	//Some SQL Statements don't return a rowset
	if(pIUnknown==NULL)
		goto CLEANUP;

	//Clear previous rowset
	ReleaseRowset();

	//[MANDATORY] Obtain [mandatory] interfaces
	XTEST(hWnd, hr = pCListBox->OutputQI(pIUnknown,	  IID_IUnknown,			(IUnknown**)&m_pIUnknown));
	XTEST(hWnd, hr = pCListBox->OutputQI(m_pIUnknown, IID_IRowset,			(IUnknown**)&m_pIRowset));
	XTEST(hWnd, hr = pCListBox->OutputQI(m_pIUnknown, IID_IAccessor,		(IUnknown**)&m_pIAccessor));
	XTEST(hWnd, hr = pCListBox->OutputQI(m_pIUnknown, IID_IColumnsInfo,		(IUnknown**)&m_pIColumnsInfo));

	//Auto QI
	if(pCOptionsDlg->m_dwRowsetOpts & ROWSET_AUTOQI)
	{
		//Obtain [optional] interfaces
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IConvertType,		(IUnknown**)&m_pIConvertType);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IRowsetInfo,		(IUnknown**)&m_pIRowsetInfo);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IColumnsRowset,	(IUnknown**)&m_pIColumnsRowset);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IConnectionPointContainer, (IUnknown**)&m_pIConnectionPointContainer);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IDBAsynchStatus,	(IUnknown**)&m_pIDBAsynchStatus);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IRowsetChange,	(IUnknown**)&m_pIRowsetChange);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IRowsetIdentity,	(IUnknown**)&m_pIRowsetIdentity);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IRowsetLocate,	(IUnknown**)&m_pIRowsetLocate);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IRowsetFind,		(IUnknown**)&m_pIRowsetFind);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IRowsetView,		(IUnknown**)&m_pIRowsetView);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IChapteredRowset,	(IUnknown**)&m_pIChapteredRowset);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IRowsetResynch,	(IUnknown**)&m_pIRowsetResynch);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IRowsetScroll,	(IUnknown**)&m_pIRowsetScroll);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_IRowsetUpdate,	(IUnknown**)&m_pIRowsetUpdate);
		hrOptional = pCListBox->OutputQI(m_pIUnknown, IID_ISupportErrorInfo,(IUnknown**)&m_pISupportErrorInfo);
	}

	if(m_pIRowsetScroll)
	{
		//Don't return a failure if this optional interface fails...
		pCListBox->OutputPreMethod("IRowsetScroll::GetApproximatePosition(DB_NULL_HCHAPTER, 0, NULL, NULL, &%d)", m_ulEstimatedRows);
		XTEST(hWnd, hrOptional = m_pIRowsetScroll->GetApproximatePosition(DB_NULL_HCHAPTER, 0, NULL, NULL, &m_ulEstimatedRows));
		pCListBox->OutputPostMethod(hrOptional, "IRowsetScroll::GetApproximatePosition(DB_NULL_HCHAPTER, 0, NULL, NULL, &%d)", m_ulEstimatedRows);
	}

	//Listeners
	AdviseListeners();

	//Setup RowPosition Object
	m_pCRowPos->CreateRowPos(m_pIUnknown);

CLEANUP:
	return hr;    
}



/////////////////////////////////////////////////////////////////
//	HRESULT CRowset::CreateDataset
//
//	called for ROWSET_FROMCOMMANDDATASET only
//
/////////////////////////////////////////////////////////////////
HRESULT CRowset::CreateDataset(ROWSETSOURCE eRowsetSource, LONG* pcRowsAffected, WCHAR* pwszCmd)
{
	ASSERT(pwszCmd);

	HRESULT	hr = S_OK;				// HRESULT
	HRESULT	hrOptional = S_OK;		// HRESULT
	HWND	hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	// initialize returned row count
	if (pcRowsAffected)
		*pcRowsAffected = DB_COUNTUNAVAILABLE;

	// Save rowset source
	m_eRowsetSource = eRowsetSource;

	ASSERT(eRowsetSource == ROWSET_FROMCOMMANDDATASET);

	// Clear previous rowset
	ReleaseRowset();

 	// Set CommandText
	m_pCDataset->ReleaseDataset();
	TESTC(hr = m_pCCommand->SetCommandText(pwszCmd));
 
 	//Execute
	XTESTC(hWnd, hr = m_pCCommand->m_pICommandText->Execute(NULL, IID_IMDDataset, NULL, NULL,
														(IUnknown**)&m_pCDataset->m_pIMDDataset));
	TESTC(pCListBox->OutputPostMethod(hr, "ICommandText::Execute(NULL, IID_IMDDataset, NULL,  &0x%08x)", m_pCDataset->m_pIMDDataset));

CLEANUP:
	// Display Property Errors (if occurred)
	if (hr == DB_S_ERRORSOCCURRED || hr == DB_E_ERRORSOCCURRED)
	{
		if (eRowsetSource == ROWSET_FROMCOMMANDDATASET)
			DisplayPropErrors(NULL, COMMAND, m_pCCommand->m_pICommandText);
	}
	return hr;    
}


	 
 /////////////////////////////////////////////////////////////////
// CMultipleResults::CMultipleResults
//
/////////////////////////////////////////////////////////////////
CMultipleResults::CMultipleResults(CMDIChild* pCMDIChild) : CBase(pCMDIChild)
{
	//MultipleResults
	m_pIMultipleResults				= NULL;//MultipleResults Interface
	m_pISupportErrorInfo			= NULL;//MultipleResults Interface
}


/////////////////////////////////////////////////////////////////
// CMultipleResults::~CMultipleResults
//
/////////////////////////////////////////////////////////////////
CMultipleResults::~CMultipleResults()
{
	ReleaseObject();
}


/////////////////////////////////////////////////////////////////
// HRESULT CMultipleResults::SetInterface
//
/////////////////////////////////////////////////////////////////
HRESULT CMultipleResults::SetInterface(REFIID riid, IUnknown* pIUnknown)
{
	HRESULT hr = S_OK;

	//no-op
	if(pIUnknown == NULL)
		return E_INVALIDARG;

	//If the requested interface is a valid interface, place it in the
	//Appropiate member variable, otherwise we have to release it, since we
	//have no place to store it...
	if(riid == IID_IUnknown) 
		m_pIUnknown = (IUnknown*)pIUnknown;
	else if(riid == IID_IMultipleResults) 
		m_pIMultipleResults = (IMultipleResults*)pIUnknown;
	else if(riid == IID_ISupportErrorInfo) 
		m_pISupportErrorInfo = (ISupportErrorInfo*)pIUnknown;
	else
		hr = E_NOINTERFACE;

	return hr;
}


/////////////////////////////////////////////////////////////////
// CMultipleResults::ReleaseRowset
//
/////////////////////////////////////////////////////////////////
HRESULT CMultipleResults::ReleaseObject()
{
	//MultipleResults
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	pCListBox->OutputRelease((IUnknown**)&m_pIMultipleResults,  "IMultipleResults");
	pCListBox->OutputRelease((IUnknown**)&m_pISupportErrorInfo,	"IRowsetErrorInfo");

	//Base
	ReleaseBase("MultipleResults");
	return S_OK;
}


////////////////////////////////////////////////////////////////
// CMultipleResults::CreateObject
//
/////////////////////////////////////////////////////////////////
HRESULT CMultipleResults::CreateObject(IUnknown* pIUnknown)
{
	HRESULT hr = E_FAIL;
	HRESULT hrOptional = E_FAIL;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	COptionsDlg* pCOptionsDlg = GetOptionsObj();

	//Some don't return a rowset
	if(pIUnknown==NULL)
		goto CLEANUP;

	//Release any previous object
	ReleaseObject();

	//Now use the new object
	//[MANDATORY]
	XTESTC(hWnd, hr = pCListBox->OutputQI(pIUnknown, IID_IUnknown,			(IUnknown**)&m_pIUnknown));
	XTESTC(hWnd, hr = pCListBox->OutputQI(pIUnknown, IID_IMultipleResults,	(IUnknown**)&m_pIMultipleResults));
	
	//Auto QI
	if(pCOptionsDlg->m_dwRowsetOpts & ROWSET_AUTOQI)
	{
		//Obtain [optional] interfaces
		hrOptional = pCListBox->OutputQI(pIUnknown, IID_ISupportErrorInfo,	(IUnknown**)&m_pISupportErrorInfo);
	}

CLEANUP:
	return hr;
}


////////////////////////////////////////////////////////////////
// CMultipleResults::GetResult
//
/////////////////////////////////////////////////////////////////
HRESULT CMultipleResults::GetResult(REFIID riid, LONG* pcRowsAffected, IUnknown** ppIUnknown)
{
	ASSERT(pcRowsAffected);
	ASSERT(ppIUnknown);

	HRESULT hr = E_FAIL;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	//No-op
	if(m_pIMultipleResults == NULL)
		return E_FAIL;
	
	//IMultipleResults::GetResult...
	TESTC(pCListBox->OutputPreMethod("IMultipleResults::GetResult(NULL, 0, %s, &%d, &0x%08x)", GetInterfaceName(riid), *pcRowsAffected, *ppIUnknown));
	XTEST(hWnd, hr = m_pIMultipleResults->GetResult(NULL, 0, riid, pcRowsAffected, ppIUnknown));
	TESTC(pCListBox->OutputPostMethod(hr, "IMultipleResults::GetResult(NULL, 0, %s, &%d, &0x%08x)", GetInterfaceName(riid), *pcRowsAffected, *ppIUnknown));

CLEANUP:
	return hr;
}


/////////////////////////////////////////////////////////////////
// CRowPos::CRowPos
//
/////////////////////////////////////////////////////////////////
CRowPos::CRowPos(CMDIChild* pCMDIChild) : CBase(pCMDIChild)
{
	//RowPosition
	m_pIConnectionPointContainer	= NULL;//RowPosition Interface
	m_pIRowPosition					= NULL;//RowPosition Interface

	//Extra Interfaces
	m_pIConnectionPoint				= NULL;//Connection Interface

	//Listeners
	m_pCRowPosChange	= new CRowPosChange(m_pCMDIChild->m_pCListBox);
}


/////////////////////////////////////////////////////////////////
// CRowPos::~CRowPos
//
/////////////////////////////////////////////////////////////////
CRowPos::~CRowPos()
{
	ReleaseRowPos();

	//Verify Listeners have no extra references on them...
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	pCListBox->OutputRelease((IUnknown**)&m_pCRowPosChange, "IRowPositionChange", 0);
}


/////////////////////////////////////////////////////////////////
// HRESULT CRowPos::SetInterface
//
/////////////////////////////////////////////////////////////////
HRESULT CRowPos::SetInterface(REFIID riid, IUnknown* pIUnknown)
{
	HRESULT hr = S_OK;

	//no-op
	if(pIUnknown == NULL)
		return E_INVALIDARG;

	//If the requested interface is a RowPosition interface, place it in the
	//Appropiate member variable, otherwise we have to release it, since we
	//have no place to store it...
	if(riid == IID_IUnknown) 
		m_pIUnknown = (IUnknown*)pIUnknown;
	else if(riid == IID_IConnectionPointContainer) 
		m_pIConnectionPointContainer = (IConnectionPointContainer*)pIUnknown;
	else if(riid == IID_IRowPosition) 
		m_pIRowPosition = (IRowPosition*)pIUnknown;
	else
		hr = E_NOINTERFACE;

	return hr;
}


/////////////////////////////////////////////////////////////////
// CRowPos::ReleaseRowPos
//
/////////////////////////////////////////////////////////////////
HRESULT CRowPos::ReleaseRowPos()
{
	//Rowset
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	pCListBox->OutputRelease((IUnknown**)&m_pIConnectionPointContainer, "IConnectionPointContainer");
	pCListBox->OutputRelease((IUnknown**)&m_pIRowPosition,				"IRowPosition");

	//Extra intefaces
	pCListBox->OutputRelease((IUnknown**)&m_pIConnectionPoint,			"IConnectionPoint");

	//Base
	ReleaseBase("RowPosition");

	//Don't Unadvise the Lsiteners until the Rowset goes away.
	//We want to receive and Display the ROWSET_RELEASE notifications
	UnadviseListeners();
	return S_OK;
}


////////////////////////////////////////////////////////////////
// CRowPos::CreateRowPos
//
/////////////////////////////////////////////////////////////////
HRESULT CRowPos::CreateRowPos(IUnknown* pIRowset)
{
	ASSERT(pIRowset);
	HRESULT hr = E_FAIL;
	HRESULT hrOptional = E_FAIL;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	//Release all previous interfaces
	ReleaseRowPos();
	
	//Create RowPosition object
	//May not have ROWPOSITIONLIBRARY installed
	TESTC(pCListBox->OutputPreMethod("CoCreateInstance(CLSID_OLEDB_ROWPOSITIONLIBRARY, NULL, CLSCTX_INPROC_SERVER, IID_IUnknown, &0x%08x", m_pIUnknown));
	hr = CoCreateInstance(CLSID_OLEDB_ROWPOSITIONLIBRARY, NULL, CLSCTX_INPROC_SERVER, IID_IUnknown, (void**)&m_pIUnknown);
	TESTC(pCListBox->OutputPostMethod(hr, "CoCreateInstance(CLSID_OLEDB_ROWPOSITIONLIBRARY, NULL, CLSCTX_INPROC_SERVER, IID_IUnknown, &0x%08x", m_pIUnknown));

	//[MANDATORY]
	XTESTC(hWnd, hr = pCListBox->OutputQI(m_pIUnknown,		IID_IRowPosition,	(IUnknown**)&m_pIRowPosition));

	//AutoQI
	if(GetOptionsObj()->m_dwRowsetOpts & ROWSET_AUTOQI)
	{
		//[OPTIONAL]
		hrOptional = pCListBox->OutputQI(m_pIUnknown,	IID_IConnectionPointContainer,	(IUnknown**)&m_pIConnectionPointContainer);
	}

	//Initialize
	TESTC(pCListBox->OutputPreMethod("IRowPosition::Initialize(0x%08x)", pIRowset));
	XTEST(hWnd, hr = m_pIRowPosition->Initialize(pIRowset));
	TESTC(pCListBox->OutputPostMethod(hr, "IRowPosition::Initialize(0x%08x)", pIRowset));

	//Advise Listeners
	AdviseListeners();
	
CLEANUP:
	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CRowPos::AdviseListeners
//
/////////////////////////////////////////////////////////////////
HRESULT CRowPos::AdviseListeners()
{
	HRESULT hr = E_FAIL;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	
	//IRowPosChange Listener
	if(m_pIRowPosition && GetOptionsObj()->m_dwNotifyOpts & NOTIFY_IROWPOSITIONCHANGE)
		TESTC(hr = m_pCRowPosChange->Advise(m_pIRowPosition));

CLEANUP:
	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CRowPos::UnadviseListeners
//
/////////////////////////////////////////////////////////////////
HRESULT CRowPos::UnadviseListeners()
{
	HRESULT hr = E_FAIL;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	
	//IRowPosChange Listener
	if(m_pIRowPosition)
		TESTC(hr = m_pCRowPosChange->Unadvise(m_pIRowPosition));

CLEANUP:
	return hr;
}


/////////////////////////////////////////////////////////////////
// CDataset::CDataset
//
/////////////////////////////////////////////////////////////////
CDataset::CDataset(CMDIChild* pCMDIChild) : CBase(pCMDIChild)
{
	//Dataset
	m_pIAccessor	= NULL;
	m_pIColumnsInfo	= NULL;
	m_pIConvertType	= NULL;
	m_pIMDDataset	= NULL;
	m_pISupportErrorInfo = NULL;

	//Data
	m_cAxis = 0;
	m_lCurAxis = 0;
	m_lCurCell = 0;
	m_rgAxisInfo = NULL;
}


/////////////////////////////////////////////////////////////////
// CDataset::~CDataset
//
/////////////////////////////////////////////////////////////////
CDataset::~CDataset()
{
	ReleaseDataset();
}


/////////////////////////////////////////////////////////////////
// ReleaseDataset
//
/////////////////////////////////////////////////////////////////
HRESULT CDataset::ReleaseDataset(void)
{
	HRESULT hr = S_OK;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	if(m_rgAxisInfo) 
	{
		hr = m_pIMDDataset->FreeAxisInfo(m_cAxis, m_rgAxisInfo);
		pCListBox->OutputPostMethod(hr, "IMDDataset::FreeAxisInfo(%d, &0x%08x)", m_cAxis, m_rgAxisInfo);
	}
	m_cAxis = 0;
	m_lCurAxis = 0;
	m_rgAxisInfo = NULL;

	//Interface
	pCListBox->OutputRelease((IUnknown**)&m_pIMDDataset,		"IMDDataSet");
	pCListBox->OutputRelease((IUnknown**)&m_pIAccessor,			"IAccessor");
	pCListBox->OutputRelease((IUnknown**)&m_pIColumnsInfo,		"IColumnsInfo");
	pCListBox->OutputRelease((IUnknown**)&m_pIConvertType,		"IConvertType");
	pCListBox->OutputRelease((IUnknown**)&m_pISupportErrorInfo,	"ISupportErrorInfo");

	//Base
	ReleaseBase("DataSet");
	return hr;
}


/////////////////////////////////////////////////////////////////
// HRESULT CDataset::SetInterface
//
/////////////////////////////////////////////////////////////////
HRESULT CDataset::SetInterface(REFIID riid, IUnknown* pIUnknown)
{
	HRESULT hr = S_OK;

	//no-op
	if(pIUnknown == NULL)
		return E_INVALIDARG;

	//If the requested interface is a RowPosition interface, place it in the
	//Appropiate member variable, otherwise we have to release it, since we
	//have no place to store it...
	if(riid == IID_IUnknown) 
		m_pIUnknown = (IUnknown*)pIUnknown;
	else if(riid == IID_IColumnsInfo) 
		m_pIColumnsInfo = (IColumnsInfo*)pIUnknown;
	else if(riid == IID_IAccessor) 
		m_pIAccessor = (IAccessor*)pIUnknown;
	else if(riid == IID_IConvertType) 
		m_pIConvertType = (IConvertType*)pIUnknown;
	else if(riid == IID_IMDDataset) 
		m_pIMDDataset = (IMDDataset*)pIUnknown;
	else if(riid == IID_ISupportErrorInfo) 
		m_pISupportErrorInfo = (ISupportErrorInfo*)pIUnknown;
	else		 
		hr = E_NOINTERFACE;

	return hr;
}


/////////////////////////////////////////////////////////////////
//	GetAxisInfo
//
//	Get the axis info for all axis in the current dataset.
//	Return count and array of axis info.
//
/////////////////////////////////////////////////////////////////
HRESULT	CDataset::GetAxisInfo(ULONG* pcAxis, MDAXISINFO** pprgAxisInfo)
{
	HRESULT hr = S_OK;
	ASSERT(pcAxis && pprgAxisInfo);
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	
	if(m_pIMDDataset==NULL)
		return E_FAIL;

	if(m_rgAxisInfo == NULL)
	{
		TESTC(hr = m_pIMDDataset->GetAxisInfo(&m_cAxis, &m_rgAxisInfo));
		TESTC(pCListBox->OutputPostMethod(hr, "IMDDataset::GetAxisInfo(%d, &0x%08x)", m_cAxis, m_rgAxisInfo));
	}
	
	*pcAxis = m_cAxis;
	*pprgAxisInfo = m_rgAxisInfo;

CLEANUP:
	return hr;    
}


/////////////////////////////////////////////////////////////////
//	GetAxisRowset
//
//	Get the axis info and rowset for the first axis.
//	Return IRowset interface for the axis rowset.
//
/////////////////////////////////////////////////////////////////
HRESULT	CDataset::GetAxisRowset(REFIID riid, IUnknown** ppIUnknown)
{
	HRESULT hr = E_FAIL;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	if(m_pIMDDataset == NULL)
		return E_FAIL;

	//GetAxisInfo
	TESTC(hr = GetAxisInfo(&m_cAxis, &m_rgAxisInfo));
		
	//GetAxisRowset
	if (m_cAxis)
	{
		TESTC(hr = m_pIMDDataset->GetAxisRowset(NULL, m_lCurAxis, riid, 0, NULL, ppIUnknown));
		TESTC(pCListBox->OutputPostMethod(hr, "IMDDataset::GetAxisRowset(NULL, 0, %s, NULL,  &0x%08x)", GetInterfaceName(riid), ppIUnknown ? *ppIUnknown : NULL));
	}

CLEANUP:
	return hr;    
}


/////////////////////////////////////////////////////////////////
//	GetIMDDataset
//
//	Get the IMDDataset interface for a mdx statement.
//
/////////////////////////////////////////////////////////////////
HRESULT	CDataset::GetIMDDataset(REFIID riid, IUnknown** ppIUnknown)
{
	HRESULT hr = S_OK;
	*ppIUnknown = NULL;
	ULONG cAxisTuple, cSliceCoord, iAxis, iOffset = 1;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	if(m_pIMDDataset == NULL)
		return E_FAIL;

	//Obtain the Specified interface
	TESTC(hr = pCListBox->OutputQI(m_pIMDDataset, riid, ppIUnknown));

	// Fetch the axis info to compute the coordinates
	TESTC(hr = GetAxisInfo(&m_cAxis, &m_rgAxisInfo ));

	// Calculate the coordinate "offset factors".
	// These are used to translate from a point in cell space
	// to a set of axis coordinates.
	//
	// Low-order Axis is contiguous.
	// High-order Axis is not contiguous.
	//
	// Example, consider cube {x,y,z} = {3,4,5}
	//   rgAxisOffset[2] = 3*4
	//   rgAxisOffset[1] = 3
	//   rgAxisOffset[0] = 1
	// Thus, where p represents the cell's ordinal value
	//   z = p / 12
	//   y = (p - z*12) / 3
	//   x = (p - z*12 - y*3)
	// And,
	//   p = x + 3*y + 12*z

	// for all axis, excluding slicer axis
	for (iAxis=0; iAxis < m_cAxis-1; iAxis++)
	{
		m_rgAxisOffset[iAxis] = iOffset;
		cAxisTuple = m_rgAxisInfo[iAxis].cCoordinates;
		iOffset *= cAxisTuple;
	}
	m_lMaxCoord = iOffset;
	cSliceCoord = m_rgAxisOffset[m_cAxis-1];

CLEANUP:
	return hr;
}


/////////////////////////////////////////////////////////////////
//	GetNextCells
//
//	Overrides pIRowset->GetNextRows()
//
/////////////////////////////////////////////////////////////////
HRESULT	CDataset::GetNextCells(LONG cRows, ULONG* pcRowsObtained, HROW* prghRows)
{
	HRESULT hr = S_OK;
	ULONG	absRows = labs(cRows);
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	*pcRowsObtained = 0;
	if (cRows > 0)
	{
		*pcRowsObtained = min(absRows, m_lMaxCoord - m_lCurCell);
	}
	else if (cRows < 0)
	{
		*pcRowsObtained = min(m_lCurCell, absRows);
		m_lCurCell -= *pcRowsObtained;
	}
	for (ULONG iRow = 0; iRow < *pcRowsObtained; iRow++)
	{
		prghRows[iRow] = m_lCurCell + iRow;
	}
	if (cRows > 0)
	{
		m_lCurCell += *pcRowsObtained;
	}
	if (*pcRowsObtained == 0) hr = DB_S_ENDOFROWSET;
	return hr;
}

/////////////////////////////////////////////////////////////////
//	GetCellData
//
//	Fetch a single cell, hRow overriden to represent the cell coordinate
//	Overrides pIRowset->GetData()
//
/////////////////////////////////////////////////////////////////
HRESULT	CDataset::GetCellData(HROW hRow, HACCESSOR hAccessor, void* pData)
{
	HRESULT hr = S_OK;
	HWND hWnd = m_pCMDIChild->m_hWnd;
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;

	if(m_pIMDDataset == NULL)
		return E_FAIL;

	//GetCellData
	XTEST(hWnd, hr = m_pIMDDataset->GetCellData(hAccessor, hRow, hRow, pData));
	TESTC(pCListBox->OutputPostMethod(hr, "IMDDataset:GetCellData(0x%08x, %d, %d, 0x%08x)", hAccessor, hRow, hRow, pData));

CLEANUP:
	return hr;
}
